<?php
require_once 'config.php';

if(isLoggedIn()) {
    header("Location: index.php");
    exit();
}

$error = '';
$success = '';

if($_SERVER['REQUEST_METHOD'] == 'POST') {
    $username = sanitize($_POST['username']);
    $email = sanitize($_POST['email']);
    $password = $_POST['password'];
    $confirm_password = $_POST['confirm_password'];
    $referral_code = isset($_POST['referral_code']) ? sanitize($_POST['referral_code']) : '';
    
    // Validation
    if(empty($username) || empty($email) || empty($password)) {
        $error = "All fields are required";
    } elseif($password != $confirm_password) {
        $error = "Passwords do not match";
    } elseif(strlen($password) < 6) {
        $error = "Password must be at least 6 characters";
    } else {
        // Check if username exists
        $check_user = $conn->prepare("SELECT id FROM users WHERE username = ? OR email = ?");
        $check_user->bind_param("ss", $username, $email);
        $check_user->execute();
        $check_user->store_result();
        
        if($check_user->num_rows > 0) {
            $error = "Username or email already exists";
        } else {
            // Generate unique referral code
            $new_referral_code = generateReferralCode();
            
            // Hash password
            $hashed_password = password_hash($password, PASSWORD_DEFAULT);
            
            // Insert user
            $stmt = $conn->prepare("INSERT INTO users (username, email, password, referral_code, referred_by) VALUES (?, ?, ?, ?, ?)");
            $stmt->bind_param("sssss", $username, $email, $hashed_password, $new_referral_code, $referral_code);
            
            if($stmt->execute()) {
                $user_id = $stmt->insert_id;
                
                // If referred by someone, create referral record
                if(!empty($referral_code)) {
                    $find_referrer = $conn->prepare("SELECT id FROM users WHERE referral_code = ?");
                    $find_referrer->bind_param("s", $referral_code);
                    $find_referrer->execute();
                    $referrer_result = $find_referrer->get_result();
                    
                    if($referrer_result->num_rows > 0) {
                        $referrer = $referrer_result->fetch_assoc();
                        $conn->query("INSERT INTO referrals (referrer_id, referred_id) VALUES ({$referrer['id']}, $user_id)");
                    }
                }
                
                $success = "Registration successful! Please login.";
                header("refresh:2;url=login.php");
            } else {
                $error = "Registration failed. Please try again.";
            }
        }
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Register - Project Sale</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/animate.css/4.1.1/animate.min.css">
</head>
<body class="bg-gradient-to-br from-blue-50 to-indigo-100 min-h-screen flex items-center justify-center p-4">
    <div class="max-w-md w-full">
        <div class="bg-white rounded-2xl shadow-2xl overflow-hidden animate__animated animate__fadeInUp">
            <!-- Header -->
            <div class="bg-gradient-to-r from-blue-600 to-indigo-600 p-8 text-center">
                <h1 class="text-3xl font-bold text-white mb-2">
                    <i class="fas fa-user-plus mr-2"></i> Create Account
                </h1>
                <p class="text-blue-100">Join our premium marketplace</p>
            </div>
            
            <!-- Form -->
            <div class="p-8">
                <?php if($error): ?>
                    <div class="bg-red-50 border-l-4 border-red-500 p-4 mb-6 animate__animated animate__shakeX">
                        <div class="flex">
                            <div class="flex-shrink-0">
                                <i class="fas fa-exclamation-circle text-red-500"></i>
                            </div>
                            <div class="ml-3">
                                <p class="text-red-700"><?php echo $error; ?></p>
                            </div>
                        </div>
                    </div>
                <?php endif; ?>
                
                <?php if($success): ?>
                    <div class="bg-green-50 border-l-4 border-green-500 p-4 mb-6 animate__animated animate__fadeIn">
                        <div class="flex">
                            <div class="flex-shrink-0">
                                <i class="fas fa-check-circle text-green-500"></i>
                            </div>
                            <div class="ml-3">
                                <p class="text-green-700"><?php echo $success; ?></p>
                            </div>
                        </div>
                    </div>
                <?php endif; ?>
                
                <form method="POST" action="" class="space-y-6">
                    <div class="relative">
                        <label class="block text-gray-700 text-sm font-bold mb-2" for="username">
                            <i class="fas fa-user text-blue-500 mr-2"></i> Username
                        </label>
                        <input type="text" id="username" name="username" required
                               class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition duration-300"
                               placeholder="Choose a username">
                    </div>
                    
                    <div class="relative">
                        <label class="block text-gray-700 text-sm font-bold mb-2" for="email">
                            <i class="fas fa-envelope text-blue-500 mr-2"></i> Email Address
                        </label>
                        <input type="email" id="email" name="email" required
                               class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition duration-300"
                               placeholder="your@email.com">
                    </div>
                    
                    <div class="relative">
                        <label class="block text-gray-700 text-sm font-bold mb-2" for="password">
                            <i class="fas fa-lock text-blue-500 mr-2"></i> Password
                        </label>
                        <input type="password" id="password" name="password" required
                               class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition duration-300"
                               placeholder="At least 6 characters">
                    </div>
                    
                    <div class="relative">
                        <label class="block text-gray-700 text-sm font-bold mb-2" for="confirm_password">
                            <i class="fas fa-lock text-blue-500 mr-2"></i> Confirm Password
                        </label>
                        <input type="password" id="confirm_password" name="confirm_password" required
                               class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition duration-300"
                               placeholder="Re-enter password">
                    </div>
                    
                    <div class="relative">
                        <label class="block text-gray-700 text-sm font-bold mb-2" for="referral_code">
                            <i class="fas fa-user-friends text-blue-500 mr-2"></i> Referral Code (Optional)
                        </label>
                        <input type="text" id="referral_code" name="referral_code"
                               class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition duration-300"
                               placeholder="Enter referral code if any">
                        <p class="text-gray-500 text-xs mt-2">You'll get special benefits if referred</p>
                    </div>
                    
                    <div class="flex items-center">
                        <input id="terms" type="checkbox" required
                               class="h-4 w-4 text-blue-600 focus:ring-blue-500 border-gray-300 rounded">
                        <label for="terms" class="ml-2 block text-sm text-gray-700">
                            I agree to the <a href="#" class="text-blue-600 hover:underline">Terms & Conditions</a>
                        </label>
                    </div>
                    
                    <button type="submit"
                            class="w-full bg-gradient-to-r from-blue-600 to-indigo-600 text-white font-bold py-3 px-4 rounded-lg hover:from-blue-700 hover:to-indigo-700 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:ring-offset-2 transition duration-300 transform hover:scale-105">
                        <i class="fas fa-user-plus mr-2"></i> Create Account
                    </button>
                </form>
                
                <div class="mt-8 pt-8 border-t border-gray-200 text-center">
                    <p class="text-gray-600">
                        Already have an account?
                        <a href="login.php" class="text-blue-600 font-semibold hover:underline ml-1">
                            <i class="fas fa-sign-in-alt mr-1"></i> Login here
                        </a>
                    </p>
                </div>
            </div>
        </div>
        
        <div class="mt-8 text-center">
            <p class="text-gray-600 text-sm">
                By registering, you agree to our platform's policies and terms.
            </p>
        </div>
    </div>
    
    <script>
        document.addEventListener('DOMContentLoaded', function() {
            // Password match validation
            const password = document.getElementById('password');
            const confirmPassword = document.getElementById('confirm_password');
            
            function validatePassword() {
                if(password.value !== confirmPassword.value) {
                    confirmPassword.style.borderColor = 'red';
                } else {
                    confirmPassword.style.borderColor = 'green';
                }
            }
            
            password.addEventListener('input', validatePassword);
            confirmPassword.addEventListener('input', validatePassword);
        });
    </script>
</body>
</html>